!>
!! @file test_redist_collection_f.f90
!! @brief Fortran test of redist_collection class
!!
!! @copyright Copyright  (C)  2016 Jörg Behrens <behrens@dkrz.de>
!!                                 Moritz Hanke <hanke@dkrz.de>
!!                                 Thomas Jahns <jahns@dkrz.de>
!!
!! @author Jörg Behrens <behrens@dkrz.de>
!!         Moritz Hanke <hanke@dkrz.de>
!!         Thomas Jahns <jahns@dkrz.de>
!!

!
! Keywords:
! Maintainer: Jörg Behrens <behrens@dkrz.de>
!             Moritz Hanke <hanke@dkrz.de>
!             Thomas Jahns <jahns@dkrz.de>
! URL: https://doc.redmine.dkrz.de/yaxt/html/
!
! Redistribution and use in source and binary forms, with or without
! modification, are  permitted provided that the following conditions are
! met:
!
! Redistributions of source code must retain the above copyright notice,
! this list of conditions and the following disclaimer.
!
! Redistributions in binary form must reproduce the above copyright
! notice, this list of conditions and the following disclaimer in the
! documentation and/or other materials provided with the distribution.
!
! Neither the name of the DKRZ GmbH nor the names of its contributors
! may be used to endorse or promote products derived from this software
! without specific prior written permission.
!
! THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
! IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
! TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
! PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
! OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
! EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
! PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
! PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
! LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
! NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
! SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
!

PROGRAM test_redist_collection
  USE mpi
  USE ftest_common, ONLY: init_mpi, finish_mpi, test_abort
  USE test_idxlist_utils, ONLY: test_err_count
  USE yaxt, ONLY: xt_initialize, xt_finalize, &
       xt_xmap, xt_xmap_delete, &
       xt_redist, xt_redist_p2p_new, xt_redist_collection_new, &
       xt_redist_delete, xt_redist_copy, &
       xt_redist_s_exchange, &
       xt_slice_c_loc
  ! older PGI compilers do not handle generic interface correctly
#if defined __PGI && (__PGIC__ < 12 || (__PGIC__ ==  12 && __PGIC_MINOR__ <= 10))
  USE xt_redist_base, ONLY: xt_redist_s_exchange
#endif
  USE test_redist_common, ONLY: build_odd_selection_xmap, check_redist
  USE iso_c_binding, ONLY: c_loc, c_ptr
  USE redist_collection_displace, ONLY: test_displacement_variations
  IMPLICIT NONE
  CALL init_mpi
  CALL xt_initialize(mpi_comm_world)

  CALL simple_test
  CALL simple_test2
  CALL test_repeated_redist(-1)
  CALL test_repeated_redist(0)
  CALL test_displacement_variations

  IF (test_err_count() /= 0) &
       CALL test_abort("non-zero error count!", &
       __FILE__, &
       __LINE__)
  CALL xt_finalize
  CALL finish_mpi
CONTAINS
  SUBROUTINE simple_test
    ! general test with one redist
    ! set up data
    TYPE(xt_xmap) :: xmap
    TYPE(xt_redist) :: redist, redist_coll, redist_copy
    INTEGER, PARAMETER :: src_slice_len = 5, dst_slice_len = 3
    DOUBLE PRECISION, PARAMETER :: &
         ref_dst_data(dst_slice_len) = (/ 1.0d0, 3.0d0, 5.0d0 /), &
         src_data(src_slice_len) = (/ 1.0d0, 2.0d0, 3.0d0, 4.0d0, 5.0d0 /)
    DOUBLE PRECISION :: dst_data(dst_slice_len)


    xmap = build_odd_selection_xmap(src_slice_len)

    redist = xt_redist_p2p_new(xmap, mpi_double_precision)
    CALL xt_xmap_delete(xmap)
    redist_copy = xt_redist_copy(redist)
    CALL xt_redist_delete(redist)
    redist = redist_copy

    ! generate redist_collection
    redist_coll = xt_redist_collection_new((/ redist /), 1, -1, mpi_comm_world)

    CALL xt_redist_delete(redist)

    ! test exchange
    CALL check_redist(redist_coll, src_data, dst_data, ref_dst_data)

    ! clean up
    CALL xt_redist_delete(redist_coll)
  END SUBROUTINE simple_test

  SUBROUTINE simple_test2
    ! general test with one redist
    ! set up data
    TYPE(xt_xmap) :: xmap
    TYPE(xt_redist) :: redist_coll, redist_copy, &
         redist_components(2)
    INTEGER, PARAMETER :: src_slice_len = 5, dst_slice_len = 3
    TYPE src_data_collection
      DOUBLE PRECISION :: dp(src_slice_len)
      LOGICAL :: l(src_slice_len)
    END TYPE src_data_collection
    TYPE dst_data_collection
      DOUBLE PRECISION :: dp(dst_slice_len)
      LOGICAL :: l(dst_slice_len)
    END TYPE dst_data_collection
    TYPE(src_data_collection), SAVE, TARGET :: src_data = src_data_collection(&
         (/ 1.0d0, 2.0d0, 3.0d0, 4.0d0, 5.0d0 /), &
         (/ .TRUE., .FALSE., .TRUE., .FALSE., .TRUE. /))
    TYPE(dst_data_collection), PARAMETER :: &
         ref_dst_data = dst_data_collection((/ 1.0d0, 3.0d0, 5.0d0 /), &
         (/ .TRUE., .TRUE., .TRUE. /))
    TYPE(dst_data_collection), TARGET :: dst_data
    TYPE(c_ptr) :: src_data_p(2), dst_data_p(2)

    xmap = build_odd_selection_xmap(src_slice_len)

    redist_components(1) = xt_redist_p2p_new(xmap, mpi_double_precision)
    redist_components(2) = xt_redist_p2p_new(xmap, mpi_logical)
    CALL xt_xmap_delete(xmap)

    ! generate redist_collection
    redist_coll = xt_redist_collection_new(redist_components, mpi_comm_world)
    CALL xt_redist_delete(redist_components)
    redist_copy = xt_redist_copy(redist_coll)
    CALL xt_redist_delete(redist_coll)
    redist_coll = redist_copy

    ! test exchange
    ! GNU Fortran versions up to 4.8 cannot call c_loc for type components,
    ! instant ICE, and some compilers won't create c_ptr's to LOGICALs
#if !defined(__GNUC__) || __GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ > 8)
    src_data_p(1) = C_LOC(src_data%dp)
#else
    CALL xt_slice_c_loc(src_data%dp, src_data_p(1))
#endif
#if !defined HAVE_FC_LOGICAL_INTEROP || !defined(__GNUC__) || __GNUC__ > 4 \
    || (__GNUC__ == 4 && __GNUC_MINOR__ > 8)
    CALL xt_slice_c_loc(src_data%l, src_data_p(2))
#else
    src_data_p(2) = C_LOC(src_data%l)
#endif
    dst_data%dp = -1.0d0
    dst_data%l = .FALSE.
#if !defined(__GNUC__) || __GNUC__ > 4 || (__GNUC__ == 4 && __GNUC_MINOR__ > 8)
    dst_data_p(1) = C_LOC(dst_data%dp)
#else
    CALL xt_slice_c_loc(dst_data%dp, dst_data_p(1))
#endif
#if !defined HAVE_FC_LOGICAL_INTEROP || !defined(__GNUC__) || __GNUC__ > 4 \
    || (__GNUC__ == 4 && __GNUC_MINOR__ > 8)
    CALL xt_slice_c_loc(dst_data%l, dst_data_p(2))
#else
    dst_data_p(2) = C_LOC(dst_data%l)
#endif
    CALL xt_redist_s_exchange(redist_coll, src_data_p, dst_data_p)
    IF (ANY(dst_data%l .NEQV. ref_dst_data%l)) &
         CALL test_abort("error in xt_redist_s_exchange", &
         __FILE__, &
         __LINE__)
    IF (ANY(dst_data%dp /= ref_dst_data%dp)) &
         CALL test_abort("error in xt_redist_s_exchange", &
         __FILE__, &
         __LINE__)

    ! clean up
    CALL xt_redist_delete(redist_coll)
  END SUBROUTINE simple_test2

  SUBROUTINE test_repeated_redist_ds1(redist_coll)
    TYPE(xt_redist), INTENT(in) :: redist_coll
    INTEGER :: i, j
    DOUBLE PRECISION, SAVE, TARGET :: src_data(5, 3) = RESHAPE((/&
         (DBLE(i), i = 1, 15)/), (/ 5, 3 /))
    DOUBLE PRECISION, PARAMETER :: ref_dst_data(3, 3) &
         = RESHAPE((/ ((DBLE(i + j), i = 1,5,2), j = 0,10,5) /), (/ 3, 3 /))
    DOUBLE PRECISION, TARGET :: dst_data(3, 3)
    TYPE(c_ptr) :: src_data_p(3), dst_data_p(3)
    dst_data = -1.0d0
    DO i = 1, 3
      CALL xt_slice_c_loc(src_data(:, i), src_data_p(i))
      CALL xt_slice_c_loc(dst_data(:, i), dst_data_p(i))
    END DO
    CALL xt_redist_s_exchange(redist_coll, 3, src_data_p, dst_data_p)

    IF (ANY(ref_dst_data /= dst_data)) &
         CALL test_abort("error in xt_redist_s_exchange", &
         __FILE__, &
         __LINE__)
  END SUBROUTINE test_repeated_redist_ds1

  SUBROUTINE test_repeated_redist_ds2(redist_coll)
    TYPE(xt_redist), INTENT(in) :: redist_coll
    INTEGER :: i, j
    DOUBLE PRECISION, SAVE, TARGET :: src_data(5, 3) = RESHAPE((/&
         (DBLE(i), i = 1, 15)/), (/ 5, 3 /))
    DOUBLE PRECISION, PARAMETER :: ref_dst_data(3, 3) &
         = RESHAPE((/ ((DBLE(i + j), i = 1,5,2), j = 0,10,5) /), (/ 3, 3 /))
    DOUBLE PRECISION, TARGET :: dst_data(3, 3)
    TYPE(c_ptr) :: src_data_p(3), dst_data_p(3)
    dst_data = -1.0d0
    CALL xt_slice_c_loc(src_data(:, 2), src_data_p(1))
    CALL xt_slice_c_loc(src_data(:, 1), src_data_p(2))
    CALL xt_slice_c_loc(src_data(:, 3), src_data_p(3))
    CALL xt_slice_c_loc(dst_data(:, 2), dst_data_p(1))
    CALL xt_slice_c_loc(dst_data(:, 1), dst_data_p(2))
    CALL xt_slice_c_loc(dst_data(:, 3), dst_data_p(3))
    CALL xt_redist_s_exchange(redist_coll, 3, src_data_p, dst_data_p)

    IF (ANY(ref_dst_data /= dst_data)) &
         CALL test_abort("error in xt_redist_s_exchange", &
         __FILE__, &
         __LINE__)
  END SUBROUTINE test_repeated_redist_ds2

  SUBROUTINE test_repeated_redist(cache_size)
    INTEGER, INTENT(in) :: cache_size
    ! test with one redist used three times (with two different input data
    ! displacements -> test of cache) (with default cache size)
    ! set up data
    INTEGER, PARAMETER :: num_slice = 3
    INTEGER, PARAMETER :: src_slice_len = 5
    TYPE(xt_xmap) :: xmap
    TYPE(xt_redist) :: redists(num_slice), redist_coll, redist_coll_copy

    xmap = build_odd_selection_xmap(src_slice_len)

    redists = xt_redist_p2p_new(xmap, mpi_double_precision)

    CALL xt_xmap_delete(xmap)

    ! generate redist_collection

    redist_coll = xt_redist_collection_new(redists, 3, cache_size, &
         mpi_comm_world)

    CALL xt_redist_delete(redists(1))

    ! test exchange
    CALL test_repeated_redist_ds1(redist_coll)
    ! test exchange with changed displacements
    CALL test_repeated_redist_ds2(redist_coll)
    ! test exchange with original displacements
    CALL test_repeated_redist_ds1(redist_coll)

    ! and the copy
    redist_coll_copy = xt_redist_copy(redist_coll)
    CALL xt_redist_delete(redist_coll)
    CALL test_repeated_redist_ds1(redist_coll_copy)
    ! test exchange with changed displacements
    CALL test_repeated_redist_ds2(redist_coll_copy)
    ! test exchange with original displacements
    CALL test_repeated_redist_ds1(redist_coll_copy)

    ! clean up
    CALL xt_redist_delete(redist_coll_copy)
  END SUBROUTINE test_repeated_redist

END PROGRAM test_redist_collection
!
! Local Variables:
! f90-continuation-indent: 5
! coding: utf-8
! indent-tabs-mode: nil
! show-trailing-whitespace: t
! require-trailing-newline: t
! End:
!
